/** 
  Prints a formatted Unicode string to the console output device specified by 
  ConOut defined in the EFI_SYSTEM_TABLE.

  This function prints a formatted Unicode string to the console output device 
  specified by ConOut in EFI_SYSTEM_TABLE and returns the number of Unicode 
  characters that printed to ConOut.  If the length of the formatted Unicode 
  string is greater than PcdUefiLibMaxPrintBufferSize, then only the first 
  PcdUefiLibMaxPrintBufferSize characters are sent to ConOut.
  If Format is NULL, then ASSERT().
  If Format is not aligned on a 16-bit boundary, then ASSERT().
  If gST->ConOut is NULL, then ASSERT().

  @param Format   A Null-terminated Unicode format string.
  @param ...      A Variable argument list whose contents are accessed based 
                  on the format string specified by Format.
  
  @return The number of Unicode characters printed to ConOut.

EDK2 - Location:      edk2\MdePkg\Library\UefiLib\UefiLibPrint.c
EDK2 Header Location: edk2\MdePkg\Include\Library\UefiLib.h
                      edk2\MdePkg\Library\UefiLib\UefiLibInternal.h
                      edk2\MdePkg\Include\Library\PrintLib.h <-- UnicodeVSPrint() eventually called

**/

#include "Uefi.h"

#if defined __cplusplus
extern "C" 
{
#endif

CONST UINT16 gUnicodeFileTag = EFI_UNICODE_BYTE_ORDER_MARK;

UEFILIB_API
UINTN
EFIAPI
Print (
  IN CONST CHAR16  *Format,
  ...
  )
{
  UINTN returnStatus;
  va_list myArgs;

  va_start(myArgs, Format);
  returnStatus = vwprintf(Format, myArgs);
  va_end(myArgs);

  return (returnStatus);
}

/**
  Convert one Null-terminated ASCII string to a Null-terminated 
  Unicode string and returns the Unicode string.

  This function converts the contents of the ASCII string Source to the Unicode 
  string Destination, and returns Destination.  The function terminates the 
  Unicode string Destination by appending a Null-terminator character at the end. 
  The caller is responsible to make sure Destination points to a buffer with size 
  equal or greater than ((AsciiStrLen (Source) + 1) * sizeof (CHAR16)) in bytes.
  
  If Destination is NULL, then ASSERT().
  If Destination is not aligned on a 16-bit boundary, then ASSERT().
  If Source is NULL, then ASSERT().
  If Source and Destination overlap, then ASSERT().
  If PcdMaximumAsciiStringLength is not zero, and Source contains more than 
  PcdMaximumAsciiStringLength ASCII characters not including the Null-terminator, 
  then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Source contains more than 
  PcdMaximumUnicodeStringLength ASCII characters not including the 
  Null-terminator, then ASSERT().

  @param  Source        Pointer to a Null-terminated ASCII string.
  @param  Destination   Pointer to a Null-terminated Unicode string.

  @reture Destination
  
  EDK2 - Location:	C:\Projects\TPM2\edk2\EdkCompatibilityPkg\Foundation\Library\EdkIIGlueLib\Library\BaseLib\String.c
  EDK2 - Header Location:	

**/
UEFILIB_API
CHAR16 *
EFIAPI
AsciiStrToUnicodeStr (
  IN      CONST CHAR8       	      *Source,
  OUT 	  CHAR16  	                *Destination
  )
{
  //ASSERT (Destination != NULL);
  //ASSERT (Source != NULL);

  int myCount = 0;
  size_t myLength = strlen(Source);

  myCount = mbstowcs(Destination, Source, myLength);

  if ( myCount <= myLength ) {
	  Destination[myCount] ='\0';
  }

  return Destination;
}


/**
  Function to determine if an entire string is a valid number.

  If hex it must be preceeded with a 0x or has ForceHex, set to TRUE.

  @param[in] theirString  The string to evaluate.
  @param[in] forceHex     TRUE - always assume hex.
  @param[in] stopAtSpace  TRUE to halt upon finding a space, FALSE to keep going.

  @retval TRUE        It is all numeric (dec/hex) characters.
  @retval FALSE       There is a non-numeric character.

  LLK::	Added this function as a util to methods which need to check whether a
        string is hex or decimal
**/
UEFILIB_API
BOOLEAN
EFIAPI
IsHexOrDecimalNumber (
  IN CONST CHAR16   *theirString,
  IN CONST BOOLEAN  forceHex,
  IN CONST BOOLEAN  stopAtSpace
  )
{
	BOOLEAN hex = FALSE;

    // Do stuff here
	//
	// Remove '-' in case there is one
	if ( (theirString != NULL) && (*theirString == L'-')) {
		theirString++;
	}
    // Chop off leading zeros
    while ( (theirString != NULL) && (*theirString == L'0')) {
		theirString++;
	}
	
	// Allow '0x' or '0X' but not 'x' nor 'X'
	if ( (theirString != NULL) && (*theirString == L'x' || *theirString == L'X') ) {
		// Is the character before this a 0?
		if (*(theirString -1) != L'0' ) {
			// We got an x without a leading 0
			return (FALSE);
		}

		theirString++;
		hex = TRUE;
	}
	else if (forceHex) {
		hex = TRUE;
	}

	// If we didn't change hex in the above IF ELSE block, then hex == false
	// Now walk through the String and verify that it is either hex or decimal
	for ( ; theirString != NULL && *theirString != CHAR_NULL && !(stopAtSpace && *theirString == L' ') ; theirString++){
		if (hex) {
			// Check that the digit is hex
			if (!isxdigit(*theirString)) {
				return (FALSE);
			}
		}
		else {
			// Check that the digit is decimal
			if (!isdigit(*theirString)) {
				return (FALSE);
			}
		}
	}

	return (TRUE);
}

/**
  Convert a Unicode character to numerical value.

  This internal function only deal with Unicode character
  which maps to a valid hexadecimal ASII character, i.e.
  L'0' to L'9', L'a' to L'f' or L'A' to L'F'. For other
  Unicode character, the value returned does not make sense.

  @param  Char  The character to convert.

  @return The numerical value converted.

**/
UEFILIB_API
UINTN
EFIAPI
HexCharToUintn (
  IN      CHAR16                    Char
  )
{
	if (isdigit(Char)) {
		return (Char - L'0');
	}
	
	return ( (UINTN)(10 + toupper(Char) - L'A'));
}

/**
  Convert a Null-terminated Unicode hexadecimal string to a value of type UINT64.

  This function returns a value of type UINTN by interpreting the contents
  of the Unicode string specified by String as a hexadecimal number.
  The format of the input Unicode string String is:

                  [spaces][zeros][x][hexadecimal digits].

  The valid hexadecimal digit character is in the range [0-9], [a-f] and [A-F].
  The prefix "0x" is optional. Both "x" and "X" are allowed in the "0x" prefix.
  If "x" appears in the input string, it must be prefixed with at least one 0.
  The function will ignore the pad space, which includes spaces or tab characters,
  before [zeros], [x] or [hexadecimal digit]. The running zero before [x] or
  [hexadecimal digit] will be ignored. The decoding starts after [x] or the
  first valid hexadecimal digit. Then, the function stops at the first character that is
  not a valid hexadecimal character or NULL, whichever one comes first.

  If String has only pad spaces, then zero is returned.
  If String has no leading pad spaces, leading zeros or valid hexadecimal digits,
  then zero is returned.

  @param[in]  String      A pointer to a Null-terminated Unicode string.
  @param[out] Value       Upon a successful return the value of the conversion.
  @param[in] StopAtSpace  FALSE to skip spaces.

  @retval EFI_SUCCESS             The conversion was successful.
  @retval EFI_INVALID_PARAMETER   A parameter was NULL or invalid.
  @retval EFI_DEVICE_ERROR        An overflow occured.
**/
UEFILIB_API
EFI_STATUS
EFIAPI
StrHexToUint64 (
  IN CONST CHAR16   *String,
     OUT   UINT64   *Value,
  IN CONST BOOLEAN  StopAtSpace
  )
{
  UINT64    Result;

  if (String == NULL || sizeof(String) == 0 || Value == NULL) {
    return (EFI_INVALID_PARAMETER);
  }

  //
  // Ignore the pad spaces (space or tab)
  //
  while ((*String == L' ') || (*String == L'\t')) {
    String++;
  }

  //
  // Ignore leading Zeros after the spaces
  //
  while (*String == L'0') {
    String++;
  }

  if (toupper(*String) == L'X') {
    if (*(String - 1) != L'0') {
      return 0;
    }
    //
    // Skip the 'X'
    //
    String++;
  }

  Result = 0;

  //
  // Skip spaces if requested
  //
  while (StopAtSpace && *String == L' ') {
    String++;
  }

  while (isxdigit(*String)) {
    //
    // If the Hex Number represented by String overflows according
    // to the range defined by UINTN, then ASSERT().
    //
	if ( !(Result <= ((((UINT64) ~0) - HexCharToUintn (*String)) >> 4))) {
    //if (!(Result <= (RShiftU64((((UINT64) ~0) - HexCharToUintn (*String)), 4)))) {
    //if (!(Result <= ((((UINT64) ~0) - InternalShellHexCharToUintn (*String)) >> 4))) {
      return (EFI_DEVICE_ERROR);
    }

    //Result = (LShiftU64(Result, 4));
    Result = Result << 4;
    Result += HexCharToUintn (*String);
    String++;

    //
    // stop at spaces if requested
    //
    if (StopAtSpace && *String == L' ') {
      break;
    }
  }

  *Value = Result;
  return (EFI_SUCCESS);
}

/**
  Convert a Null-terminated Unicode decimal string to a value of
  type UINT64.

  This function returns a value of type UINT64 by interpreting the contents
  of the Unicode string specified by String as a decimal number. The format
  of the input Unicode string String is:

                  [spaces] [decimal digits].

  The valid decimal digit character is in the range [0-9]. The
  function will ignore the pad space, which includes spaces or
  tab characters, before [decimal digits]. The running zero in the
  beginning of [decimal digits] will be ignored. Then, the function
  stops at the first character that is a not a valid decimal character
  or a Null-terminator, whichever one comes first.

  If String has only pad spaces, then 0 is returned.
  If String has no pad spaces or valid decimal digits,
  then 0 is returned.

  @param[in]  String      A pointer to a Null-terminated Unicode string.
  @param[out] Value       Upon a successful return the value of the conversion.
  @param[in] StopAtSpace  FALSE to skip spaces.

  @retval EFI_SUCCESS             The conversion was successful.
  @retval EFI_INVALID_PARAMETER   A parameter was NULL or invalid.
  @retval EFI_DEVICE_ERROR        An overflow occured.

LLK :: Added this as a util function in string conversions
**/
UEFILIB_API
EFI_STATUS
EFIAPI
StrDecimalToUint64 (
  IN CONST CHAR16 *String,
     OUT   UINT64 *Value,
  IN CONST BOOLEAN  StopAtSpace
  )
{
  UINT64     Result;

  if (String == NULL || StrSize (String) == 0 || Value == NULL) {
    return (EFI_INVALID_PARAMETER);
  }

  //
  // Ignore the pad spaces (space or tab)
  //
  while ((*String == L' ') || (*String == L'\t')) {
    String++;
  }

  //
  // Ignore leading Zeros after the spaces
  //
  while (*String == L'0') {
    String++;
  }

  Result = 0;

  //
  // Skip spaces if requested
  //
  while (StopAtSpace && *String == L' ') {
    String++;
  }

  while (isdigit(*String)) {
    //
    // If the number represented by String overflows according
    // to the range defined by UINT64, then ASSERT().
    //
    if (! (Result <= (((UINT64) ~0) - (*String - L'0'))/10)) {
    //if (!(Result <= (DivU64x32((((UINT64) ~0) - (*String - L'0')),10)))) {
      return (EFI_DEVICE_ERROR);
    }

    //Result = MultU64x32(Result, 10) + (*String - L'0');
    Result = (Result * 10) + (*String - L'0');
    String++;

    //
    // Stop at spaces if requested
    //
    if (StopAtSpace && *String == L' ') {
      break;
    }
  }

  *Value = Result;

  return (EFI_SUCCESS);  // We should return (EFI_SUCCESS); when it is defined
}


/**
  Function to verify and convert a string to its numerical value.

  If Hex it must be preceeded with a 0x, 0X, or has ForceHex set TRUE.

  @param[in] String       The string to evaluate.
  @param[out] Value       Upon a successful return the value of the conversion.
  @param[in] ForceHex     TRUE - always assume hex.
  @param[in] StopAtSpace  FALSE to skip spaces.

  @retval EFI_SUCCESS             The conversion was successful.
  @retval EFI_INVALID_PARAMETER   String contained an invalid character.
  @retval EFI_NOT_FOUND           String was a number, but Value was NULL.
  
  EDK2 - Location:	edk2\ShellPkg\Library\UefiShellLib\UefiShellLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
ShellConvertStringToUint64(
  IN CONST CHAR16   *String,
     OUT   UINT64   *Value,
  IN CONST BOOLEAN  ForceHex,
  IN CONST BOOLEAN  StopAtSpace
  )
{
	BOOLEAN hex = ForceHex;
	UINT64 returnValue;
	CONST CHAR16  *myChar;
	int status = 0;


	// First verify that we have a hex or decimal number string
	if (!IsHexOrDecimalNumber(String, hex, StopAtSpace)) {
		if (!hex) {
			// May have failed because ForceHex was false. Try again
			hex = TRUE;
			if (!IsHexOrDecimalNumber(String, hex, StopAtSpace)) {
				return (EFI_INVALID_PARAMETER);
			}
		} else {
			// The number string is bad
			return (EFI_INVALID_PARAMETER);
		}
	}

	// Remove leading spaces
	for (myChar = String; myChar != NULL && *myChar != CHAR_NULL && *myChar == L' '; myChar++);

	//
	// Check to make sure we have something left that is numeric
	//
	if (myChar == NULL || *myChar == CHAR_NULL || !IsHexOrDecimalNumber(myChar, hex, StopAtSpace)) {
		return (EFI_INVALID_PARAMETER);
	}

	//
	// Do the conversion
	//
	if (hex || StrnCmp(myChar, L"0x", 2) == 0 || StrnCmp(myChar, L"0X", 2) == 0){
		status = StrHexToUint64(myChar, &returnValue, StopAtSpace);
	}else {
		status = StrDecimalToUint64(myChar, &returnValue, StopAtSpace);
	}

	// Check that Value is a valid pointer
	if (Value == NULL ) { // edk2 also checks && !EFI_ERROR(Status)
		return (EFI_NOT_FOUND);
	}
	
	if (Value != NULL) {
		*Value = returnValue;
	}

	return (EFI_SUCCESS);
}

/**
  Copies one Null-terminated Unicode string to another Null-terminated Unicode
  string and returns the new Unicode string.

  This function copies the contents of the Unicode string Source to the Unicode
  string Destination, and returns Destination. If Source and Destination
  overlap, then the results are undefined.

  If Destination is NULL, then ASSERT().
  If Destination is not aligned on a 16-bit boundary, then ASSERT().
  If Source is NULL, then ASSERT().
  If Source is not aligned on a 16-bit boundary, then ASSERT().
  If Source and Destination overlap, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Source contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().

  @param  Destination A pointer to a Null-terminated Unicode string.
  @param  Source      A pointer to a Null-terminated Unicode string.

  @return Destination.

  EDK2 - Location:	edk2\MdePkg\Library\BaseLib\String.c

**/
UEFILIB_API
CHAR16 *
EFIAPI
StrCpy (
  OUT     CHAR16                    *Destination,
  IN      CONST CHAR16              *Source
  )
{
  CHAR16                            *ReturnValue;

  //
  // Destination cannot be NULL
  //
  ASSERT (Destination != NULL);
  ASSERT (((UINTN) Destination & BIT0) == 0);

  //
  // Destination and source cannot overlap
  //
  ASSERT ((UINTN)(Destination - Source) > StrLen (Source));
  ASSERT ((UINTN)(Source - Destination) > StrLen (Source));

  ReturnValue = wcscpy(Destination, Source);

  return ReturnValue;
}

/**
  Copies up to a specified length from one Null-terminated Unicode string  to 
  another Null-terminated Unicode string and returns the new Unicode string.

  This function copies the contents of the Unicode string Source to the Unicode
  string Destination, and returns Destination. At most, Length Unicode
  characters are copied from Source to Destination. If Length is 0, then
  Destination is returned unmodified. If Length is greater that the number of
  Unicode characters in Source, then Destination is padded with Null Unicode
  characters. If Source and Destination overlap, then the results are
  undefined.

  If Length > 0 and Destination is NULL, then ASSERT().
  If Length > 0 and Destination is not aligned on a 16-bit boundary, then ASSERT().
  If Length > 0 and Source is NULL, then ASSERT().
  If Length > 0 and Source is not aligned on a 16-bit boundary, then ASSERT().
  If Source and Destination overlap, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Length is greater than 
  PcdMaximumUnicodeStringLength, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Source contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the Null-terminator,
  then ASSERT().

  @param  Destination A pointer to a Null-terminated Unicode string.
  @param  Source      A pointer to a Null-terminated Unicode string.
  @param  Length      The maximum number of Unicode characters to copy.

  @return Destination.

  EDK2 - Location:	edk2\MdePkg\Library\BaseLib\String.c

**/
UEFILIB_API
CHAR16 *
EFIAPI
StrnCpy (
  OUT     CHAR16                    *Destination,
  IN      CONST CHAR16              *Source,
  IN      UINTN                     Length
  )
{
  CHAR16                            *ReturnValue;

  if (Length == 0) {
    return Destination;
  }

  //
  // Destination cannot be NULL if Length is not zero
  //
  ASSERT (Destination != NULL);
  ASSERT (((UINTN) Destination & BIT0) == 0);

  //
  // Destination and source cannot overlap
  //
  ASSERT ((UINTN)(Destination - Source) > StrLen (Source));
  ASSERT ((UINTN)(Source - Destination) >= Length);
  /** LLK commented out the IF block
  if (PcdGet32 (PcdMaximumUnicodeStringLength) != 0) {
    ASSERT (Length <= PcdGet32 (PcdMaximumUnicodeStringLength));
  }
  **/

  ReturnValue = wcsncpy(Destination, Source, Length);

  return ReturnValue;
}

/**
  Compares two Null-terminated Unicode strings, and returns the difference
  between the first mismatched Unicode characters.

  This function compares the Null-terminated Unicode string FirstString to the
  Null-terminated Unicode string SecondString. If FirstString is identical to
  SecondString, then 0 is returned. Otherwise, the value returned is the first
  mismatched Unicode character in SecondString subtracted from the first
  mismatched Unicode character in FirstString.

  If FirstString is NULL, then ASSERT().
  If FirstString is not aligned on a 16-bit boundary, then ASSERT().
  If SecondString is NULL, then ASSERT().
  If SecondString is not aligned on a 16-bit boundary, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and FirstString contains more
  than PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and SecondString contains more
  than PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().

  @param  FirstString   A pointer to a Null-terminated Unicode string.
  @param  SecondString  A pointer to a Null-terminated Unicode string.

  @retval 0      FirstString is identical to SecondString.
  @return others FirstString is not identical to SecondString.

**/
UEFILIB_API
INTN
EFIAPI
StrCmp (
  IN      CONST CHAR16              *FirstString,
  IN      CONST CHAR16              *SecondString
  )
{
  //
  // ASSERT both strings are less long than PcdMaximumUnicodeStringLength
  //
  ASSERT (StrSize (FirstString) != 0);
  ASSERT (StrSize (SecondString) != 0);

  return (wcscmp(FirstString, SecondString));
}

/**
  Compares up to a specified length the contents of two Null-terminated Unicode strings,
  and returns the difference between the first mismatched Unicode characters.
  
  This function compares the Null-terminated Unicode string FirstString to the
  Null-terminated Unicode string SecondString. At most, Length Unicode
  characters will be compared. If Length is 0, then 0 is returned. If
  FirstString is identical to SecondString, then 0 is returned. Otherwise, the
  value returned is the first mismatched Unicode character in SecondString
  subtracted from the first mismatched Unicode character in FirstString.

  If Length > 0 and FirstString is NULL, then ASSERT().
  If Length > 0 and FirstString is not aligned on a 16-bit boundary, then ASSERT().
  If Length > 0 and SecondString is NULL, then ASSERT().
  If Length > 0 and SecondString is not aligned on a 16-bit boundary, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Length is greater than
  PcdMaximumUnicodeStringLength, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and FirstString contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the Null-terminator,
  then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and SecondString contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the Null-terminator,
  then ASSERT().

  @param  FirstString   A pointer to a Null-terminated Unicode string.
  @param  SecondString  A pointer to a Null-terminated Unicode string.
  @param  Length        The maximum number of Unicode characters to compare.

  @retval 0      FirstString is identical to SecondString.
  @return others FirstString is not identical to SecondString.

**/
UEFILIB_API
INTN
EFIAPI
StrnCmp (
  IN      CONST CHAR16              *FirstString,
  IN      CONST CHAR16              *SecondString,
  IN      UINTN                     Length
  )
{
  if (Length == 0) {
    return 0;
  }

  //
  // ASSERT both strings are less long than PcdMaximumUnicodeStringLength.
  // Length tests are performed inside StrLen().
  //
  ASSERT (StrSize (FirstString) != 0);
  ASSERT (StrSize (SecondString) != 0);
  /** LLK: Not sure what this is but ... commenting it out
  if (PcdGet32 (PcdMaximumUnicodeStringLength) != 0) {
    ASSERT (Length <= PcdGet32 (PcdMaximumUnicodeStringLength));
  }
  **/


  return (wcsncmp(FirstString, SecondString, Length));
}


/**
  Returns the length of a Null-terminated Unicode string.

  This function returns the number of Unicode characters in the Null-terminated
  Unicode string specified by String.

  If String is NULL, then ASSERT().
  If String is not aligned on a 16-bit boundary, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and String contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().

  @param  String  A pointer to a Null-terminated Unicode string.

  @return The length of String.

  EDK2 - Location:	edk2\MdePkg\Library\BaseLib\String.c

**/
UEFILIB_API
UINTN
EFIAPI
StrLen (
  IN      CONST CHAR16              *String
  )
{

  ASSERT (String != NULL);
  ASSERT (((UINTN) String & BIT0) == 0);

  return (wcslen(String));
}


/**
  Concatenates one Null-terminated Unicode string to another Null-terminated
  Unicode string, and returns the concatenated Unicode string.

  This function concatenates two Null-terminated Unicode strings. The contents
  of Null-terminated Unicode string Source are concatenated to the end of
  Null-terminated Unicode string Destination. The Null-terminated concatenated
  Unicode String is returned. If Source and Destination overlap, then the
  results are undefined.

  If Destination is NULL, then ASSERT().
  If Destination is not aligned on a 16-bit boundary, then ASSERT().
  If Source is NULL, then ASSERT().
  If Source is not aligned on a 16-bit boundary, then ASSERT().
  If Source and Destination overlap, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Destination contains more
  than PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and Source contains more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and concatenating Destination
  and Source results in a Unicode string with more than
  PcdMaximumUnicodeStringLength Unicode characters, not including the
  Null-terminator, then ASSERT().

  @param  Destination A pointer to a Null-terminated Unicode string.
  @param  Source      A pointer to a Null-terminated Unicode string.

  @return Destination.

  EDK2 - Location:	edk2\MdePkg\Library\BaseLib\String.c

**/
UEFILIB_API
CHAR16 *
EFIAPI
StrCat (
  IN OUT  CHAR16                    *Destination,
  IN      CONST CHAR16              *Source
  )
{

  wcscat(Destination, Source);
  //
  // Size of the resulting string should never be zero.
  // PcdMaximumUnicodeStringLength is tested inside StrLen().
  //
  ASSERT (StrSize (Destination) != 0);

  return Destination;
}

/**
  Returns the first occurrence of a Null-terminated Unicode sub-string
  in a Null-terminated Unicode string.

  This function scans the contents of the Null-terminated Unicode string
  specified by String and returns the first occurrence of SearchString.
  If SearchString is not found in String, then NULL is returned.  If
  the length of SearchString is zero, then String is
  returned.

  If String is NULL, then ASSERT().
  If String is not aligned on a 16-bit boundary, then ASSERT().
  If SearchString is NULL, then ASSERT().
  If SearchString is not aligned on a 16-bit boundary, then ASSERT().

  If PcdMaximumUnicodeStringLength is not zero, and SearchString
  or String contains more than PcdMaximumUnicodeStringLength Unicode
  characters, not including the Null-terminator, then ASSERT().

  @param  String          A pointer to a Null-terminated Unicode string.
  @param  SearchString    A pointer to a Null-terminated Unicode string to search for.

  @retval NULL            If the SearchString does not appear in String.
  @return others          If there is a match.

**/
UEFILIB_API
CHAR16 *
EFIAPI
StrStr (
  IN      CONST CHAR16              *String,
  IN      CONST CHAR16              *SearchString
  )
{

  //
  // Verify we do not have bogus strings
  //
  //ASSERT (StrSize (String) != 0);
  //ASSERT (StrSize (SearchString) != 0);

  return (wcsstr(String, SearchString));
}

/**
  Function to take a formatted argument and print it to a file.

  @param[in] Handle   the file handle for the file to write to
  @param[in] Format   the format argument (see printlib for format specifier)
  @param[in] ...      the variable arguments for the format

  @retval EFI_SUCCESS the operation was sucessful
  @return other       a return value from FileHandleWriteLine

  @sa FileHandleWriteLine
  
  EDK2 - Location:	edk2\ShellPkg\Library\UefiFileHandleLib\UefiFileHandleLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
FileHandlePrintLine(
  IN EFI_FILE_HANDLE  Handle,
  IN CONST CHAR16     *Format,
  ...
  )
{
  va_list           vl;
  const int				bufferSize = 4096;
  CHAR16            buffer[4096];
  EFI_STATUS        status;

  //
  // Clear buffer first
  //
  memset(buffer, 0, (bufferSize * sizeof(CHAR16)));
  ASSERT(buffer != NULL);

  //
  // Print into our buffer
  //
  va_start(vl, Format);
  vswprintf(buffer, bufferSize, Format, vl);
  va_end(vl);

  //
  // Print buffer into file
  //
  status = FileHandleWriteLine(Handle, buffer);

  return EFI_SUCCESS;
}


/**
  Function to read a single line (up to but not including the \n) from a EFI_FILE_HANDLE.

  If the position upon start is 0, then the Ascii Boolean will be set.  This should be
  maintained and not changed for all operations with the same file.

  @param[in]       Handle        FileHandle to read from
  @param[in, out]  Buffer        pointer to buffer to read into
  @param[in, out]  Size          pointer to number of bytes in buffer
  @param[in]       Truncate      if TRUE then allows for truncation of the line to fit.
                                 if FALSE will reset the position to the begining of the
                                 line if the buffer is not large enough.
  @param[in, out]  Ascii         Boolean value for indicating whether the file is Ascii (TRUE) or UCS2 (FALSE);

  @retval EFI_SUCCESS           the operation was sucessful.  the line is stored in
                                Buffer.
  @retval EFI_INVALID_PARAMETER Handle was NULL.
  @retval EFI_INVALID_PARAMETER Size was NULL.
  @retval EFI_BUFFER_TOO_SMALL  Size was not enough space to store the line.
                                Size was updated to minimum space required.
  @sa FileHandleRead
  
  EDK2 - Location:	C:\Projects\TPM2\edk2\ShellPkg\Library\UefiFileHandleLib\UefiFileHandleLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
FileHandleReadLine(
  IN EFI_FILE_HANDLE            Handle,
  IN OUT CHAR16                 *Buffer,
  IN OUT UINTN                  *Size,
  IN BOOLEAN                    Truncate,
  IN OUT BOOLEAN                *Ascii
  )
{
  EFI_STATUS  Status;
  CHAR16      CharBuffer;
  UINTN       CharSize;
  UINTN       CountSoFar;
  fpos_t      OriginalFilePosition;
  int c;

  if (Handle == NULL
    ||Size   == NULL
	||*Ascii  == TRUE	//Bill
    ||Truncate == FALSE	//Bill

   ){
    return (EFI_INVALID_PARAMETER);
  }
  if (Buffer == NULL) {
    ASSERT(*Size == 0);
  } else {
    *Buffer = CHAR_NULL;
  }

  /*
  fgetpos(Handle, (fpos_t *)&OriginalFilePosition);
  Print(L"ftell: %d, 0x%x\n", _ftelli64(Handle),_ftelli64(Handle));	//Bill
  if (_ftelli64(Handle) == 0) {										//Bill
//if (OriginalFilePosition == 0) {
    CharSize = sizeof(CHAR16);
    Status = fread(&CharBuffer, CharSize, 1, Handle);
	ASSERT(Status == 1);
    if (CharBuffer == gUnicodeFileTag) {
      *Ascii = FALSE;
    } else {
      *Ascii = TRUE;
	  fsetpos(Handle, (fpos_t *)&OriginalFilePosition);
    }
  }
  */
  for (CountSoFar = 0;;CountSoFar++){
	  /*
    CharBuffer = 0;
    if (*Ascii) {
      CharSize = sizeof(CHAR8);
    } else {
      CharSize = sizeof(CHAR16);
    }
    Status = fread(&CharBuffer, CharSize, 1, Handle);
	Status = (Status == 1) ? EFI_SUCCESS : EFI_DEVICE_ERROR; // set the status correctly
    if (  Status != EFI_SUCCESS
       || CharSize == 0
       || (CharBuffer == L'\n' && !(*Ascii))
       || (CharBuffer ==  '\n' && *Ascii)
     ){
      break;
	  */
	  //Bill
	CharBuffer = 0;
	CharSize = sizeof(CHAR16);
	Status = fread(&CharBuffer, CharSize, 1, Handle);
	Status = (Status == 1) ? EFI_SUCCESS : EFI_DEVICE_ERROR; // set the status correctly
	if (  Status != EFI_SUCCESS
		|| CharSize == 0
		|| (CharBuffer ==  L'\n')
		){
		break;
    }
    //
    // if we have space save it...
    //
	if (CharBuffer != gUnicodeFileTag) {				//Bill
		if ((CountSoFar+1) * sizeof(CHAR16) < *Size){
		//if ((CountSoFar+1) < *Size){
		  ASSERT(Buffer != NULL);
		  ((CHAR16*)Buffer)[CountSoFar] = CharBuffer;
		  ((CHAR16*)Buffer)[CountSoFar+1] = CHAR_NULL;
		}
	}													//Bill
  }

  //
  // if we ran out of space tell when...
  //
  if ((CountSoFar+1) * sizeof(CHAR16) > *Size){
  //if ((CountSoFar+1) > *Size){
	  /*
    *Size = (CountSoFar+1)*sizeof(CHAR16);
    if (!Truncate) {
      fsetpos(Handle, (fpos_t *)&OriginalFilePosition);
    }
    return (EFI_BUFFER_TOO_SMALL);
	*/
	return (EFI_BUFFER_TOO_SMALL);						//Bill
  }
  while(StrLen(Buffer) > 0 && Buffer[StrLen(Buffer)-1] == L'\r') {
    Buffer[StrLen(Buffer)-1] = CHAR_NULL;
  }
 // Print(L"ftell: %d, 0x%x\n", _ftelli64(Handle),_ftelli64(Handle));
  return (Status);
}

/**
  function to write a line of unicode text to a file.

  if Handle is NULL, ASSERT.
  if Buffer is NULL, do nothing.  (return SUCCESS)

  @param[in]     Handle         FileHandle to write to
  @param[in]     Buffer         Buffer to write

  @retval  EFI_SUCCESS          the data was written.
  @retval  other                failure.

  @sa FileHandleWrite

  EDK2 - Location:	edk2\ShellPkg\Library\UefiFileHandleLib\UefiFileHandleLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
FileHandleWriteLine(
  IN EFI_FILE_HANDLE Handle,
  IN CHAR16          *Buffer
  )
{
  EFI_STATUS Status;
  UINTN      Size;

  ASSERT(Handle != NULL);

  if (Buffer == NULL) {
    return (EFI_SUCCESS);
  }

  Size = StrSize(Buffer);
  Status = fwrite(Buffer, sizeof(CHAR16), Size, Handle);
  if (Status != Size) {
    return (Status);
  }
  Size = StrSize(L"\n");
  Status = fwrite(L"\n", sizeof(CHAR16), Size, Handle);
  if(Status == Size)
	  return EFI_SUCCESS;
  else
	  return EFI_DEVICE_ERROR;
}

/**
  This function will open a file or directory referenced by filename.

  If return is EFI_SUCCESS, the Filehandle is the opened file's handle;
  otherwise, the Filehandle is NULL. The Attributes is valid only for
  EFI_FILE_MODE_CREATE.

  if FileName is NULL then ASSERT()

  @param  FileName      pointer to file name
  @param  FileHandle    pointer to the file handle.
  @param  OpenMode      the mode to open the file with.
  @param  Attributes    the file's file attributes.

  @retval EFI_SUCCESS           The information was set.
  @retval EFI_INVALID_PARAMETER One of the parameters has an invalid value.
  @retval EFI_UNSUPPORTED       Could not open the file path.
  @retval EFI_NOT_FOUND         The specified file could not be found on the
                                device or the file system could not be found
                                on the device.
  @retval EFI_NO_MEDIA          The device has no medium.
  @retval EFI_MEDIA_CHANGED     The device has a different medium in it or the
                                medium is no longer supported.
  @retval EFI_DEVICE_ERROR      The device reported an error.
  @retval EFI_VOLUME_CORRUPTED  The file system structures are corrupted.
  @retval EFI_WRITE_PROTECTED   The file or medium is write protected.
  @retval EFI_ACCESS_DENIED     The file was opened read only.
  @retval EFI_OUT_OF_RESOURCES  Not enough resources were available to open the
                                file.
  @retval EFI_VOLUME_FULL       The volume is full.

  EDK2 - Location:	edk2\ShellPkg\Library\UefiShellLib\UefiShellLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
ShellOpenFileByName(
  IN CONST CHAR16               *FileName,
  OUT SHELL_FILE_HANDLE         *FileHandle,
  IN UINT64                     OpenMode,
  IN UINT64                     Attributes
  )
{
  ASSERT(FileName != NULL);
  ASSERT(FileHandle != NULL);
  ASSERT(OpenMode != 0);
  ASSERT(Attributes == 0); // Attributes are not used by the TPM2 Tool at this point, so they haven't been implemented

  if (FileName == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  if((OpenMode & EFI_FILE_MODE_CREATE) == EFI_FILE_MODE_CREATE)
	//*FileHandle = _wfopen(FileName, L"w+, ccs=UTF-16LE");
	*FileHandle = _wfopen(FileName, L"w+b");
  else if((OpenMode & EFI_FILE_MODE_WRITE) == EFI_FILE_MODE_WRITE)
	//*FileHandle = _wfopen(FileName, L"r+, ccs=UTF-16LE");
	*FileHandle = _wfopen(FileName, L"r+b");
  else if((OpenMode & EFI_FILE_MODE_READ) == EFI_FILE_MODE_READ)
	*FileHandle = _wfopen(FileName, L"r, ccs=UTF-16LE");
  else {
	*FileHandle = NULL;
	return EFI_INVALID_PARAMETER;
  }

  if(*FileHandle == NULL)
	return EFI_INVALID_PARAMETER;
  else
	return EFI_SUCCESS;
}

/**
  This function reads information from an opened file.

  If FileHandle is not a directory, the function reads the requested number of
  bytes from the file at the file's current position and returns them in Buffer.
  If the read goes beyond the end of the file, the read length is truncated to the
  end of the file. The file's current position is increased by the number of bytes
  returned.  If FileHandle is a directory, the function reads the directory entry
  at the file's current position and returns the entry in Buffer. If the Buffer
  is not large enough to hold the current directory entry, then
  EFI_BUFFER_TOO_SMALL is returned and the current file position is not updated.
  BufferSize is set to be the size of the buffer needed to read the entry. On
  success, the current position is updated to the next directory entry. If there
  are no more directory entries, the read returns a zero-length buffer.
  EFI_FILE_INFO is the structure returned as the directory entry.

  @param FileHandle             the opened file handle
  @param BufferSize             on input the size of buffer in bytes.  on return
                                the number of bytes written.
  @param Buffer                 the buffer to put read data into.

  @retval EFI_SUCCESS           Data was read.
  @retval EFI_NO_MEDIA          The device has no media.
  @retval EFI_DEVICE_ERROR  The device reported an error.
  @retval EFI_VOLUME_CORRUPTED  The file system structures are corrupted.
  @retval EFI_BUFFER_TO_SMALL Buffer is too small. ReadSize contains required
                                size.

**/
UEFILIB_API
EFI_STATUS
EFIAPI
ShellReadFile(
  IN SHELL_FILE_HANDLE          FileHandle,
  IN OUT UINTN                  *BufferSize,
  OUT VOID                      *Buffer
  )
{
  int sz = 0;

  if(FileHandle == NULL || Buffer == NULL || BufferSize == NULL || *BufferSize == 0)
	  return EFI_INVALID_PARAMETER;

  sz = fread(Buffer, 1, *BufferSize, FileHandle);
  if(sz == *BufferSize)
  {
	  *BufferSize = sz;
	  return EFI_SUCCESS;
  }
  else
  {
	  return EFI_LOAD_ERROR;
  }
}

/**
  Close an open file handle.

  This function closes a specified file handle. All "dirty" cached file data is
  flushed to the device, and the file is closed. In all cases the handle is
  closed.

@param FileHandle               the file handle to close.

@retval EFI_SUCCESS             the file handle was closed sucessfully.

EDK2 - Location:	edk2\ShellPkg\Library\UefiShellLib\UefiShellLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
ShellCloseFile (
  IN SHELL_FILE_HANDLE                     *FileHandle
  )
{
  if(*FileHandle == NULL)
	return EFI_INVALID_PARAMETER;

  if(fclose(*FileHandle) == 0)
	return EFI_SUCCESS;
  else
	return EFI_INVALID_PARAMETER;
}

/** @file
  ZeroMem() implementation.

  The following BaseMemoryLib instances contain the same copy of this file:

    BaseMemoryLib
    BaseMemoryLibMmx
    BaseMemoryLibSse2
    BaseMemoryLibRepStr
    BaseMemoryLibOptDxe
    BaseMemoryLibOptPei
    PeiMemoryLib
    UefiMemoryLib
    
  Copyright (c) 2006 - 2010, Intel Corporation. All rights reserved.<BR>
  This program and the accompanying materials
  are licensed and made available under the terms and conditions of the BSD License
  which accompanies this distribution.  The full text of the license may be found at
  http://opensource.org/licenses/bsd-license.php.

  THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
  WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

//#include "MemLibInternals.h"

/**
  Fills a target buffer with zeros, and returns the target buffer.

  This function fills Length bytes of Buffer with zeros, and returns Buffer.
  
  If Length > 0 and Buffer is NULL, then ASSERT().
  If Length is greater than (MAX_ADDRESS - Buffer + 1), then ASSERT().

  @param  Buffer      The pointer to the target buffer to fill with zeros.
  @param  Length      The number of bytes in Buffer to fill with zeros.

  @return Buffer.

EDK2 - Location:	edk2\MdePkg\Library\BaseMemoryLibOptDxe\ZeroMemWrapper.c
**/
UEFILIB_API
VOID *
EFIAPI
ZeroMem (
  OUT VOID  *Buffer,
  IN UINTN  Length
  )
{
  ASSERT (!(Buffer == NULL && Length > 0));
  ASSERT (Length <= (MAX_ADDRESS - (UINTN)Buffer + 1));

  return memset(Buffer, 0, Length);
}


/** @file
  CopyMem() implementation.

  The following BaseMemoryLib instances contain the same copy of this file:
  
    BaseMemoryLib
    BaseMemoryLibMmx
    BaseMemoryLibSse2
    BaseMemoryLibRepStr
    BaseMemoryLibOptDxe
    BaseMemoryLibOptPei
    PeiMemoryLib
    UefiMemoryLib

  Copyright (c) 2006 - 2010, Intel Corporation. All rights reserved.<BR>
  This program and the accompanying materials
  are licensed and made available under the terms and conditions of the BSD License
  which accompanies this distribution.  The full text of the license may be found at
  http://opensource.org/licenses/bsd-license.php.

  THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
  WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

//#include "MemLibInternals.h"

/**
  Copies a source buffer to a destination buffer, and returns the destination buffer.

  This function copies Length bytes from SourceBuffer to DestinationBuffer, and returns
  DestinationBuffer.  The implementation must be reentrant, and it must handle the case
  where SourceBuffer overlaps DestinationBuffer.
  
  If Length is greater than (MAX_ADDRESS - DestinationBuffer + 1), then ASSERT().
  If Length is greater than (MAX_ADDRESS - SourceBuffer + 1), then ASSERT().

  @param  DestinationBuffer   The pointer to the destination buffer of the memory copy.
  @param  SourceBuffer        The pointer to the source buffer of the memory copy.
  @param  Length              The number of bytes to copy from SourceBuffer to DestinationBuffer.

  @return DestinationBuffer.
  
  edk2 - Location:	edk2\MdePkg\Library\BaseMemoryLibOptDxe\CopyMemWrapper.c

**/
UEFILIB_API
VOID *
EFIAPI
CopyMem (
  OUT VOID       *DestinationBuffer,
  IN CONST VOID  *SourceBuffer,
  IN UINTN       Length
  )
{
  
  if (Length == 0) {
    return DestinationBuffer;
  }
  ASSERT ((Length - 1) <= (MAX_ADDRESS - (UINTN)DestinationBuffer));
  ASSERT ((Length - 1) <= (MAX_ADDRESS - (UINTN)SourceBuffer));

  if (DestinationBuffer == SourceBuffer) {
    return DestinationBuffer;
  }
  
  return memcpy(DestinationBuffer, SourceBuffer, Length);
}


/**
  Write data to a file.

  This function writes the specified number of bytes to the file at the current
  file position. The current file position is advanced the actual number of bytes
  written, which is returned in BufferSize. Partial writes only occur when there
  has been a data error during the write attempt (such as "volume space full").
  The file is automatically grown to hold the data if required. Direct writes to
  opened directories are not supported.

  @param FileHandle           The opened file for writing
  @param BufferSize           on input the number of bytes in Buffer.  On output
                              the number of bytes written.
  @param Buffer               the buffer containing data to write is stored.

 @retval EFI_SUCCESS          Data was written.
 @retval EFI_UNSUPPORTED      Writes to an open directory are not supported.
 @retval EFI_NO_MEDIA         The device has no media.
 @retval EFI_DEVICE_ERROR     The device reported an error.
 @retval EFI_VOLUME_CORRUPTED The file system structures are corrupted.
 @retval EFI_WRITE_PROTECTED  The device is write-protected.
 @retval EFI_ACCESS_DENIED    The file was open for read only.
 @retval EFI_VOLUME_FULL      The volume is full.
 
 edk2 - Location:	edk2\ShellPkg\Library\UefiShellLib\UefiShellLib.c
**/
UEFILIB_API
EFI_STATUS
EFIAPI
ShellWriteFile(
  IN SHELL_FILE_HANDLE          FileHandle,
  IN OUT UINTN                  *BufferSize,
  IN VOID                       *Buffer
  )
{
  int ret = EFI_SUCCESS;

  int bytesWritten = fwrite(Buffer, sizeof(BYTE), *BufferSize, FileHandle);
  
  if(bytesWritten != *BufferSize)
  {
	  int error = ferror(FileHandle);
	  // TODO: map this error code to EFI error code
	  ret = -1;
  }
  

  *BufferSize = bytesWritten;
  
  return ret;
}


/**
  Produces a Null-terminated Unicode string in an output buffer based on a Null-terminated 
  Unicode format string and variable argument list.
  
  Produces a Null-terminated Unicode string in the output buffer specified by StartOfBuffer
  and BufferSize.
  The Unicode string is produced by parsing the format string specified by FormatString.
  Arguments are pulled from the variable argument list based on the contents of the format string.
  The number of Unicode characters in the produced output buffer is returned not including
  the Null-terminator.
  If BufferSize is 0 or 1, then no output buffer is produced and 0 is returned.

  If BufferSize > 1 and StartOfBuffer is NULL, then ASSERT().
  If BufferSize > 1 and StartOfBuffer is not aligned on a 16-bit boundary, then ASSERT().
  If BufferSize > 1 and FormatString is NULL, then ASSERT().
  If BufferSize > 1 and FormatString is not aligned on a 16-bit boundary, then ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and FormatString contains more than 
  PcdMaximumUnicodeStringLength Unicode characters not including the Null-terminator, then
  ASSERT().
  If PcdMaximumUnicodeStringLength is not zero, and produced Null-terminated Unicode string
  contains more than PcdMaximumUnicodeStringLength Unicode characters not including the
  Null-terminator, then ASSERT().

  @param  StartOfBuffer   A pointer to the output buffer for the produced Null-terminated 
                          Unicode string.
  @param  BufferSize      The size, in bytes, of the output buffer specified by StartOfBuffer.
  @param  FormatString    A Null-terminated Unicode format string.
  @param  ...             Variable argument list whose contents are accessed based on the 
                          format string specified by FormatString.
  
  @return The number of Unicode characters in the produced output buffer not including the
          Null-terminator.
          
  edk2 - Location:	C:\Projects\TPM2\edk2\MdePkg\Library\BasePrintLib\PrintLib.c
**/
UEFILIB_API
UINTN
EFIAPI
UnicodeSPrint (
  OUT CHAR16        *StartOfBuffer,
  IN  UINTN         BufferSize,
  IN  CONST CHAR16  *FormatString,
  ...
  )
{
  UINTN   NumberOfPrinted = 0;

  va_list vl;
  va_start(vl, FormatString);

  NumberOfPrinted = vswprintf(StartOfBuffer, BufferSize, FormatString, vl);

  va_end(vl);

  return NumberOfPrinted;
}

/**
  Frees a buffer that was previously allocated with one of the pool allocation functions in the
  Memory Allocation Library.

  Frees the buffer specified by Buffer.  Buffer must have been allocated on a previous call to the
  pool allocation services of the Memory Allocation Library.  If it is not possible to free pool
  resources, then this function will perform no actions.
  
  If Buffer was not allocated with a pool allocation function in the Memory Allocation Library,
  then ASSERT().

  @param  Buffer                The pointer to the buffer to free.
  
  EDK2 - Location:	edk2\MdePkg\Library\UefiMemoryAllocationLib\MemoryAllocationLib.c

**/
UEFILIB_API
VOID
EFIAPI
FreePool (
  IN VOID   *Buffer
  )
{
  free(Buffer);
}


/**
  Allocates and zeros a buffer of type EfiBootServicesData.

  Allocates the number bytes specified by AllocationSize of type EfiBootServicesData, clears the
  buffer with zeros, and returns a pointer to the allocated buffer.  If AllocationSize is 0, then a
  valid buffer of 0 size is returned.  If there is not enough memory remaining to satisfy the
  request, then NULL is returned.

  @param  AllocationSize        The number of bytes to allocate and zero.

  @return A pointer to the allocated buffer or NULL if allocation fails.
  
  EDK2 - Location:	edk2\MdePkg\Library\UefiMemoryAllocationLib\MemoryAllocationLib.c

**/
UEFILIB_API
VOID *
EFIAPI
AllocateZeroPool (
  IN UINTN  AllocationSize
  )
{
  VOID *buffer = malloc(AllocationSize);
  memset(buffer, 0, AllocationSize);
  return buffer;
}


/**
  Compares the contents of two buffers.

  This function compares Length bytes of SourceBuffer to Length bytes of DestinationBuffer.
  If all Length bytes of the two buffers are identical, then 0 is returned.  Otherwise, the
  value returned is the first mismatched byte in SourceBuffer subtracted from the first
  mismatched byte in DestinationBuffer.
  
  If Length > 0 and DestinationBuffer is NULL, then ASSERT().
  If Length > 0 and SourceBuffer is NULL, then ASSERT().
  If Length is greater than (MAX_ADDRESS - DestinationBuffer + 1), then ASSERT().
  If Length is greater than (MAX_ADDRESS - SourceBuffer + 1), then ASSERT().

  @param  DestinationBuffer The pointer to the destination buffer to compare.
  @param  SourceBuffer      The pointer to the source buffer to compare.
  @param  Length            The number of bytes to compare.

  @return 0                 All Length bytes of the two buffers are identical.
  @retval Non-zero          The first mismatched byte in SourceBuffer subtracted from the first
                            mismatched byte in DestinationBuffer.
                            
**/
UEFILIB_API
INTN
EFIAPI
CompareMem (
  IN CONST VOID  *DestinationBuffer,
  IN CONST VOID  *SourceBuffer,
  IN UINTN       Length
  )
{
	if (Length == 0 || DestinationBuffer == SourceBuffer) {
		return 0;
	}
	ASSERT (DestinationBuffer != NULL);
	ASSERT (SourceBuffer != NULL);
	ASSERT ((Length - 1) <= (MAX_ADDRESS - (UINTN)DestinationBuffer));
	ASSERT ((Length - 1) <= (MAX_ADDRESS - (UINTN)SourceBuffer));

	return InternalMemCompareMem (DestinationBuffer, SourceBuffer, Length);
}


/**
  Compares two memory buffers of a given length.

  @param  DestinationBuffer The first memory buffer
  @param  SourceBuffer      The second memory buffer
  @param  Length            The length of DestinationBuffer and SourceBuffer memory
                            regions to compare. Must be non-zero.

  @return 0                 All Length bytes of the two buffers are identical.
  @retval Non-zero          The first mismatched byte in SourceBuffer subtracted from the first
                            mismatched byte in DestinationBuffer.

**/
UEFILIB_API
INTN
EFIAPI
InternalMemCompareMem (
  IN      CONST VOID                *DestinationBuffer,
  IN      CONST VOID                *SourceBuffer,
  IN      UINTN                     Length
  )
{
  while ((--Length != 0) &&
         (*(INT8*)DestinationBuffer == *(INT8*)SourceBuffer)) {
    DestinationBuffer = (INT8*)DestinationBuffer + 1;
    SourceBuffer = (INT8*)SourceBuffer + 1;
  }
  return (INTN)*(UINT8*)DestinationBuffer - (INTN)*(UINT8*)SourceBuffer;
}


/**
  Retrieve the size of a file.

  This function extracts the file size info from the FileHandle's EFI_FILE_INFO
  data.

  @param[in] FileHandle         The file handle from which size is retrieved.
  @param[out] Size              The pointer to size.

  @retval EFI_SUCCESS           The operation was completed sucessfully.
  @retval EFI_DEVICE_ERROR      Cannot access the file.
**/

UEFILIB_API
EFI_STATUS
EFIAPI
ShellGetFileSize (
  IN SHELL_FILE_HANDLE          FileHandle,
  OUT UINT64                    *Size
  )
{
	fseek(FileHandle, 0L, SEEK_END);
	*Size = ftell(FileHandle);
	rewind(FileHandle);
}


UEFILIB_API
VOID
EFIAPI
Exit()
{
}

UEFILIB_API
VOID
EFIAPI
UefiSetConsoleTextColor(int color)
{
  if(color != WinConsoleColor_Black)
  {
    HANDLE h = GetStdHandle(STD_OUTPUT_HANDLE);
    unsigned short wAttributes= (unsigned int)color;
    SetConsoleTextAttribute(h,wAttributes);
  }
}


/**
  String comparison without regard to case for a limited number of characters.

  @param[in] Source   The first item to compare.
  @param[in] Target   The second item to compare.
  @param[in] Count    How many characters to compare.

  @retval NULL Source and Target are identical strings without regard to case.
  @return The location in Source where there is a difference.
**/
UEFILIB_API
CONST CHAR16*
EFIAPI
StrniCmp(
  IN CONST CHAR16 *Source,
  IN CONST CHAR16 *Target,
  IN CONST UINTN  Count
  )
{
  UINTN   LoopCount;
  CHAR16  Char1;
  CHAR16  Char2;
  CHAR16  Error[7] = L"Error\n";

  if (Source == NULL || Target == NULL) 
    return Error;

  for (LoopCount = 0 ; LoopCount < Count ; LoopCount++) {
    Char1 = towupper(Source[LoopCount]);
    Char2 = towupper(Target[LoopCount]);
    if (Char1 != Char2) {
      return (&Source[LoopCount]);
    }
  }
  return (NULL);
}






#if defined __cplusplus
}
#endif